/*
 * Copyright (C) 2017 Igalia S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include "WebKitIconLoadingClient.h"

#include "APIIconLoadingClient.h"
#include "WebKitWebViewPrivate.h"
#include <wtf/TZoneMallocInlines.h>
#include <wtf/glib/GWeakPtr.h>

using namespace WebKit;

class IconLoadingClient : public API::IconLoadingClient {
    WTF_MAKE_TZONE_ALLOCATED_INLINE(IconLoadingClient);
public:
    explicit IconLoadingClient(WebKitWebView* webView)
        : m_webView(webView)
    {
    }

private:
    void getLoadDecisionForIcon(const WebCore::LinkIcon& icon, CompletionHandler<void(CompletionHandler<void(API::Data*)>&&)>&& completionHandler) override
    {
        // WebCore can send non HTTP icons.
        if (!icon.url.protocolIsInHTTPFamily()) {
            completionHandler(nullptr);
            return;
        }

        WebCore::LinkIcon copiedIcon = icon;
        webkitWebViewGetLoadDecisionForIcon(m_webView, icon, [weakWebView = GWeakPtr<WebKitWebView>(m_webView), icon = WTFMove(copiedIcon), completionHandler = WTFMove(completionHandler)] (bool loadIcon) mutable {
            if (!weakWebView || !loadIcon) {
                completionHandler(nullptr);
                return;
            }

            completionHandler([weakWebView = WTFMove(weakWebView), icon = WTFMove(icon)] (API::Data* iconData) {
                if (!weakWebView || !iconData)
                    return;
                webkitWebViewSetIcon(weakWebView.get(), icon, *iconData);
            });
        });
    }

    WebKitWebView* m_webView;
};

void attachIconLoadingClientToView(WebKitWebView* webView)
{
    webkitWebViewGetPage(webView).setIconLoadingClient(makeUnique<IconLoadingClient>(webView));
}
